document.addEventListener('DOMContentLoaded', function () {
    const sendBtn = document.getElementById('sendBtn');
    const kindleEmailInput = document.getElementById('kindleEmail');
    const status = document.getElementById('status');
    const remainingDiv = document.getElementById('remaining');

    restoreSavedEmail().then(() => {
        // Fetch remaining emails after restoring saved email
        if (kindleEmailInput.value.trim()) {
            fetchRemaining();
        }
    });
    
    kindleEmailInput.addEventListener('change', fetchRemaining);
    kindleEmailInput.addEventListener('blur', fetchRemaining);

    sendBtn.addEventListener('click', async function () {
        const kindleEmail = kindleEmailInput.value.trim();
        if (!kindleEmail) {
            updateStatus('Please enter your Kindle email address.', 'error');
            return;
        }

        try {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

            if (tab.url && (tab.url.toLowerCase().endsWith('.pdf') || tab.url.includes('pdf'))) {
                updateStatus('Processing PDF...', 'success');
                await saveEmail(kindleEmail);

                chrome.runtime.sendMessage({
                    action: 'sendToKindle',
                    url: tab.url,
                    title: tab.title,
                    kindleEmail
                }, function (response) {
                    if (chrome.runtime.lastError) {
                        updateStatus('Error: ' + chrome.runtime.lastError.message, 'error');
                        return;
                    }

                    if (response && response.success) {
                        updateStatus('PDF sent to Kindle successfully!', 'success');
                    } else {
                        updateStatus('Failed to send PDF', 'error');
                    }
                });
            } else {
                updateStatus('Current page is not a PDF', 'error');
            }
        } catch (error) {
            updateStatus('Error: ' + error.message, 'error');
        }
    });

    function restoreSavedEmail() {
        return new Promise((resolve, reject) => {
            chrome.storage.sync.get(['kindleEmail'], (result) => {
                if (chrome.runtime.lastError) {
                    console.error('Failed to load saved Kindle email', chrome.runtime.lastError);
                    reject(chrome.runtime.lastError);
                    return;
                }

                if (result.kindleEmail) {
                    kindleEmailInput.value = result.kindleEmail;
                }
                resolve();
            });
        });
    }

    function saveEmail(email) {
        return new Promise((resolve, reject) => {
            chrome.storage.sync.set({ kindleEmail: email }, () => {
                if (chrome.runtime.lastError) {
                    reject(chrome.runtime.lastError);
                } else {
                    resolve();
                }
            });
        });
    }

    function updateStatus(message, type) {
        const statusText = document.getElementById('statusText');
        statusText.textContent = message;
        status.style.display = 'flex';
        status.className = type === 'error' ? 'alert alert-error' : 'alert alert-success';
    }

    async function fetchRemaining() {
        const kindleEmail = kindleEmailInput.value.trim();
        if (!kindleEmail) {
            remainingDiv.innerHTML = '';
            return;
        }

        try {
            const response = await fetch(`${CONFIG.API_BASE_URL}/api/remaining?kindle_email=${encodeURIComponent(kindleEmail)}`);
            if (!response.ok) {
                console.error('API error:', response.status, response.statusText);
                remainingDiv.innerHTML = '';
                return;
            }
            const data = await response.json();

            if (data.is_paying) {
                remainingDiv.innerHTML = 'Thank you for supporting PDFling.com! 🎉';
            } else {
                const remaining = data.remaining;
                remainingDiv.innerHTML = `${remaining} / ${CONFIG.MAX_FREE_EMAILS} free emails remaining (<a href="${CONFIG.LIFETIME_DEAL_URL}" class="link" target="_blank">upgrade here</a>).`;
            }
        } catch (error) {
            console.error('Failed to fetch remaining emails:', error);
            remainingDiv.innerHTML = '';
        }
    }
});
